import { Player } from "@minecraft/server";
import * as minecraftserver from '@minecraft/server';

import { ActionFormData, ModalFormData, MessageFormData, ActionFormResponse, ModalFormResponse, MessageFormResponse } from "@minecraft/server-ui";

export class interfaceActionForm
{
    /**
     * @param {String|minecraftserver.RawMessage} title 
     * @param {String|minecraftserver.RawMessage} body 
     * @param {({ player=Player.prototype, other, canceled=ActionFormResponse.prototype.canceled, cancelationReason=ActionFormResponse.prototype.cancelationReason, selection=ActionFormResponse.prototype.selection })=>{}} callback 
     */
    constructor(title="", body="", callback=({player, other, canceled, cancelationReason, selection})=>{})
    {
        this.interface= new ActionFormData();
        this.interface.title(title);
        this.interface.body(body);
        this.callback=callback;
        this.buttons=[];
    }
    /**
     * 
     * @param {String|minecraftserver.RawMessage} name 
     * @param {String} img 
     * @param {({ player=Player.prototype, other })=>{}} callback 
     * @returns 
     */
    button(name="", img="", callback=({ player, other })=>{})
    {
        this.buttons.push({name:name, img:img, click:callback});
        this.interface.button(name, img);
        return this
    }
    /**
    * @param {Player} player
    * @param {*} other
    */
    async show(player, other)
    {
        let {canceled, cancelationReason, selection} = await this.interface.show(player)
        if(!canceled)
            this.buttons[selection].click({player, other})
        this.callback({player, other, canceled, cancelationReason, selection})
    }
}
export class interfaceModalForm
{
    /**
     * @param {String|minecraftserver.RawMessage} title 
     * @param {({ player=Player.prototype, other, canceled=ModalFormResponse.prototype.canceled, cancelationReason=ModalFormResponse.prototype.cancelationReason, formValues=ModalFormResponse.prototype.formValues })=>{}} callback 
     */
    constructor(title="", callback=({ player, other, canceled, cancelationReason, formValues })=>{})
    {
        this.interface= new ModalFormData()
        this.interface.title(title);
        this.callback=callback;
        this.modals=[];
    }
    /**
     * @param {String|minecraftserver.RawMessage} label 
     * @param {String|minecraftserver.RawMessage} placeholderText 
     * @param {String} defaultValue 
     * @param {({ player=Player.prototype, other, formValue=ModalFormResponse.prototype.formValues })=>{}} callback 
     * @returns 
     */
    textField(label="", placeholderText="", defaultValue="", callback=({ player, other, formValue })=>{})
    {
        this.modals.push({label:label, placeholderText:placeholderText, defaultValue:defaultValue, click:callback});
        this.interface.textField(label, placeholderText, defaultValue);
        return this
    }
    /**
     * @param {String|minecraftserver.RawMessage} label 
     * @param {Boolean} defaultValue 
     * @param {({ player=Player.prototype, other, formValue=ModalFormResponse.prototype.formValues })=>{}} callback 
     * @returns 
     */
    toggle(label="", defaultValue=false, callback=({ player, other, formValue })=>{})
    {
        this.modals.push({label:label, defaultValue:defaultValue, click:callback});
        this.interface.toggle(label, defaultValue);
        return this
    }
    /**
     * @param {String|minecraftserver.RawMessage} label 
     * @param {[String|minecraftserver.RawMessage]} options 
     * @param {Number} defaultValueIndex 
     * @param {({ player=Player.prototype, other, formValue=ModalFormResponse.prototype.formValues })=>{}} callback 
     * @returns 
     */
    dropdown(label="", options=[""], defaultValueIndex=0, callback=({ player, other, formValue })=>{})
    {
        this.modals.push({label:label, defaultValue:defaultValueIndex, click:callback});
        this.interface.dropdown(label, options, defaultValueIndex);
        return this
    }
    /**
     * @param {String|minecraftserver.RawMessage} label 
     * @param {Number} defaultValue 
     * @param {({ player=Player.prototype, other, formValue=ModalFormResponse.prototype.formValues })=>{}} callback 
     * @returns 
     */
    slider(label="", min=0, max=1, valueStep=1, defaultValue=0, callback=({ player, other, formValue })=>{})
    {
        this.modals.push({label:label, minimumValue:min, maximumValue:max, valueStep:valueStep, defaultValue:defaultValue, click:callback});
        this.interface.slider(label, min, max, valueStep, defaultValue);
        return this
    }
    /**
    * @param {Player} player
    * @param {*} other
    */
    async show(player, other)
    {
        let {canceled, cancelationReason, formValues} = await this.interface.show(player);
        let i=0;
        if(!canceled)
            for(let formValue of formValues)
                this.modals[i++].click({player, other, formValue})
        this.callback({ player, other, canceled, cancelationReason, formValues })
        //return { player, other, canceled, cancelationReason, formValues }
    }
}
export class interfaceMessageForm{
    /**
     * 
     * @param {String|minecraftserver.RawMessage} title 
     * @param {String|minecraftserver.RawMessage} body 
     * @param {({ player=Player.prototype, other, canceled=MessageFormResponse.prototype.canceled, cancelationReason=MessageFormResponse.prototype.cancelationReason, selection=MessageFormResponse.prototype.selection })=>{}} callback 
     */
    constructor(title="", body="", callback=({ player, other, canceled, cancelationReason, selection })=>{})
    {
        this.interface= new MessageFormData()
        this.interface.title(title);
        this.interface.body(body);
        this.callback=callback;
        this.buttons=[];
    }
    /**
     * @param {String|minecraftserver.RawMessage} text 
     * @param {({ player=Player.prototype, other})=>{}} callback 
     * @returns 
     */
    button(text="", callback=({ player, other })=>{})
    {
        this.buttons.push({text:text, click:callback});
        return this
    }
    /**
    * @param {Player} player
    * @param {*} other
    */
    async show (player, other)
    {
        if(this.buttons[1]!==undefined)
            this.interface.button1(this.buttons[1].text);
        if(this.buttons[0]!==undefined)
            this.interface.button2(this.buttons[0].text);

        let {canceled, cancelationReason, selection} = await this.interface.show(player)
        if(!canceled)
            if(selection)
                this.buttons[0].click({player, other: other})
            else
                this.buttons[1].click({player, other: other})
        this.callback({player, other, canceled, cancelationReason, selection})
        //return result
    }
}
function addPrefixAndSuffixToJSON(jsonString) {
    let keyPrefix= "§c"
    let keySuffix= "§2"
    let valuePrefix= "§3"
    let valueSuffix= "§2"
    let jsonObject = JSON.parse(jsonString);
    let modifiedObject = {};
    
    for (let prop in jsonObject) {
        let modifiedKey = keyPrefix + prop + keySuffix;
        let modifiedValue = valuePrefix + jsonObject[prop] + valueSuffix;
        modifiedObject[modifiedKey] = modifiedValue;
    }

    let finalString = JSON.stringify(modifiedObject).replace(/"(.*?)":\s*"(.*?)"/g, function(match, group1, group2) {
        return `"${group1}":${group2}`;
    });
    return finalString;
}
    

/**
 * 
 * @param {String} namespace 
 * @param {String} name 
 * @param {{}} eventJSON
 * @param {String|Number|Boolean} defaultValue 
 * @param {String} valueNamespace 
 * @returns 
 */
export function ButtonNameBehaviorComponent (namespace, behavior){
    let text={rawtext:[{translate:""}]}
    text.rawtext.push({translate:`${namespace}`})
    if(behavior===undefined)
        text.rawtext.push({translate:`gc.boolean.behavior.disable`})
    else
        text.rawtext.push({translate:`gc.boolean.behavior.enable`, with:[`#${behavior.priority}`]})
    return text
}
/**
 * 
 * @param {String} namespace 
 * @param {String} name 
 * @param {{}} eventJSON
 * @param {String|Number|Boolean} defaultValue 
 * @param {String} valueNamespace 
 * @returns 
 */
export function ButtonNameTemplate (namespace, name, eventJSON, defaultValue, valueNamespace=undefined){
    let text={rawtext:[{translate:""}]}
    if(eventJSON===undefined)
    {
        text.rawtext.push({translate:`${namespace}`})
        text.rawtext.push({translate:`${namespace}.${name}`})
    }
    if(typeof(defaultValue)==="string")
    {
        text.rawtext.push({translate:`${namespace}.${name}`})
        if(valueNamespace===undefined)
            text.rawtext.push({translate:`${namespace}.${name}.${eventJSON[name]||defaultValue}`})
        else
            text.rawtext.push({translate:`${valueNamespace}${eventJSON[name]||defaultValue}`})
    }
	if(typeof(defaultValue)==="number")
    {
        text.rawtext.push({translate:`${namespace}.${name}`})
        if(valueNamespace===undefined)
            text.rawtext.push({translate:`${Number(eventJSON[name]).toString()==="NaN"?defaultValue:eventJSON[name]}`})
        else
            text.rawtext.push({translate:`${valueNamespace}${Number(eventJSON[name]).toString()==="NaN"?defaultValue:eventJSON[name]}`})

    }
	if(typeof(defaultValue)==="boolean")
    {
        text.rawtext.push({translate:`${namespace}.${name}`})
        if(valueNamespace===undefined)
            text.rawtext.push({translate:`${eventJSON[name]===false?false:eventJSON[name]===true?true:defaultValue}`})
        else
            text.rawtext.push({translate:`${valueNamespace}${eventJSON[name]===false?false:eventJSON[name]===true?true:defaultValue}`})
    }
    if(typeof(defaultValue)==="object")
    {
        text.rawtext.push({translate:`${namespace}.${name}`})
        text.rawtext.push({translate:`${addPrefixAndSuffixToJSON(JSON.stringify(eventJSON[name]||defaultValue))}`})
    }
    return text
}
/**
 * 
 * @param {{target:{type:"nothing", value:"nothing"}}} component 
 * @returns 
 */
export function ButtonNameBehaviorTargetComponent (component={target:{type:"nothing", value:"nothing"}}){
    let text={rawtext:[{translate:""}]}
    text.rawtext.push({translate:`gc.npc.interface.target`})
    switch (component.target.type) {
        case "all":
            text.rawtext.push({translate:`gc.npc.interface.target.all`})
            break;
        case "family":
            text.rawtext.push({translate:`gc.npc.interface.target.family.${component.target.value}`})
            break;
        case "tag":
            text.rawtext.push({text:`${component.target.value}`})
            break;
        default:
            text.rawtext.push({translate:`gc.npc.interface.target.nothing`})
            break;
    }
    return text
}
